/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.world.level.chunk.storage;

import com.google.common.annotations.VisibleForTesting;
import com.mojang.logging.LogUtils;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import javax.annotation.Nullable;
import net.minecraft.Util;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.profiling.jfr.JvmProfiler;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.chunk.storage.RegionBitmap;
import net.minecraft.world.level.chunk.storage.RegionFileVersion;
import net.minecraft.world.level.chunk.storage.RegionStorageInfo;
import org.slf4j.Logger;

public class RegionFile
implements AutoCloseable {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final int SECTOR_BYTES = 4096;
    @VisibleForTesting
    protected static final int SECTOR_INTS = 1024;
    private static final int CHUNK_HEADER_SIZE = 5;
    private static final int HEADER_OFFSET = 0;
    private static final ByteBuffer PADDING_BUFFER = ByteBuffer.allocateDirect(1);
    private static final String EXTERNAL_FILE_EXTENSION = ".mcc";
    private static final int EXTERNAL_STREAM_FLAG = 128;
    private static final int EXTERNAL_CHUNK_THRESHOLD = 256;
    private static final int CHUNK_NOT_PRESENT = 0;
    final RegionStorageInfo info;
    private final Path path;
    private final FileChannel file;
    private final Path externalFileDir;
    final RegionFileVersion version;
    private final ByteBuffer header = ByteBuffer.allocateDirect(8192);
    private final IntBuffer offsets;
    private final IntBuffer timestamps;
    @VisibleForTesting
    protected final RegionBitmap usedSectors = new RegionBitmap();

    public RegionFile(RegionStorageInfo regionstorageinfo, Path path, Path path1, boolean flag) throws IOException {
        this(regionstorageinfo, path, path1, RegionFileVersion.getSelected(), flag);
    }

    public RegionFile(RegionStorageInfo regionstorageinfo, Path path, Path path1, RegionFileVersion regionfilecompression, boolean flag) throws IOException {
        this.info = regionstorageinfo;
        this.path = path;
        this.version = regionfilecompression;
        if (!Files.isDirectory(path1, new LinkOption[0])) {
            throw new IllegalArgumentException("Expected directory, got " + String.valueOf(path1.toAbsolutePath()));
        }
        this.externalFileDir = path1;
        this.offsets = this.header.asIntBuffer();
        ((Buffer)this.offsets).limit(1024);
        ((Buffer)this.header).position(4096);
        this.timestamps = this.header.asIntBuffer();
        this.file = flag ? FileChannel.open(path, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.DSYNC) : FileChannel.open(path, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        this.usedSectors.force(0, 2);
        ((Buffer)this.header).position(0);
        int i = this.file.read(this.header, 0L);
        if (i != -1) {
            if (i != 8192) {
                LOGGER.warn("Region file {} has truncated header: {}", (Object)path, (Object)i);
            }
            long j = Files.size(path);
            for (int k = 0; k < 1024; ++k) {
                int l = this.offsets.get(k);
                if (l == 0) continue;
                int i1 = RegionFile.getSectorNumber(l);
                int j1 = RegionFile.getNumSectors(l);
                if (i1 < 2) {
                    LOGGER.warn("Region file {} has invalid sector at index: {}; sector {} overlaps with header", new Object[]{path, k, i1});
                    this.offsets.put(k, 0);
                    continue;
                }
                if (j1 == 0) {
                    LOGGER.warn("Region file {} has an invalid sector at index: {}; size has to be > 0", (Object)path, (Object)k);
                    this.offsets.put(k, 0);
                    continue;
                }
                if ((long)i1 * 4096L > j) {
                    LOGGER.warn("Region file {} has an invalid sector at index: {}; sector {} is out of bounds", new Object[]{path, k, i1});
                    this.offsets.put(k, 0);
                    continue;
                }
                this.usedSectors.force(i1, j1);
            }
        }
    }

    public Path getPath() {
        return this.path;
    }

    private Path getExternalChunkPath(ChunkPos chunkcoordintpair) {
        String s = "c." + chunkcoordintpair.x + "." + chunkcoordintpair.z + EXTERNAL_FILE_EXTENSION;
        return this.externalFileDir.resolve(s);
    }

    @Nullable
    public synchronized DataInputStream getChunkDataInputStream(ChunkPos chunkcoordintpair) throws IOException {
        int i = this.getOffset(chunkcoordintpair);
        if (i == 0) {
            return null;
        }
        int j = RegionFile.getSectorNumber(i);
        int k = RegionFile.getNumSectors(i);
        int l = k * 4096;
        ByteBuffer bytebuffer = ByteBuffer.allocate(l);
        this.file.read(bytebuffer, j * 4096);
        ((Buffer)bytebuffer).flip();
        if (bytebuffer.remaining() < 5) {
            LOGGER.error("Chunk {} header is truncated: expected {} but read {}", new Object[]{chunkcoordintpair, l, bytebuffer.remaining()});
            return null;
        }
        int i1 = bytebuffer.getInt();
        byte b0 = bytebuffer.get();
        if (i1 == 0) {
            LOGGER.warn("Chunk {} is allocated, but stream is missing", (Object)chunkcoordintpair);
            return null;
        }
        int j1 = i1 - 1;
        if (RegionFile.isExternalStreamChunk(b0)) {
            if (j1 != 0) {
                LOGGER.warn("Chunk has both internal and external streams");
            }
            return this.createExternalChunkInputStream(chunkcoordintpair, RegionFile.getExternalChunkVersion(b0));
        }
        if (j1 > bytebuffer.remaining()) {
            LOGGER.error("Chunk {} stream is truncated: expected {} but read {}", new Object[]{chunkcoordintpair, j1, bytebuffer.remaining()});
            return null;
        }
        if (j1 < 0) {
            LOGGER.error("Declared size {} of chunk {} is negative", (Object)i1, (Object)chunkcoordintpair);
            return null;
        }
        JvmProfiler.INSTANCE.onRegionFileRead(this.info, chunkcoordintpair, this.version, j1);
        return this.createChunkInputStream(chunkcoordintpair, b0, RegionFile.createStream(bytebuffer, j1));
    }

    private static int getTimestamp() {
        return (int)(Util.getEpochMillis() / 1000L);
    }

    private static boolean isExternalStreamChunk(byte b0) {
        return (b0 & 0x80) != 0;
    }

    private static byte getExternalChunkVersion(byte b0) {
        return (byte)(b0 & 0xFFFFFF7F);
    }

    @Nullable
    private DataInputStream createChunkInputStream(ChunkPos chunkcoordintpair, byte b0, InputStream inputstream) throws IOException {
        RegionFileVersion regionfilecompression = RegionFileVersion.fromId(b0);
        if (regionfilecompression == RegionFileVersion.VERSION_CUSTOM) {
            String s = new DataInputStream(inputstream).readUTF();
            ResourceLocation minecraftkey = ResourceLocation.tryParse(s);
            if (minecraftkey != null) {
                LOGGER.error("Unrecognized custom compression {}", (Object)minecraftkey);
                return null;
            }
            LOGGER.error("Invalid custom compression id {}", (Object)s);
            return null;
        }
        if (regionfilecompression == null) {
            LOGGER.error("Chunk {} has invalid chunk stream version {}", (Object)chunkcoordintpair, (Object)b0);
            return null;
        }
        return new DataInputStream(regionfilecompression.wrap(inputstream));
    }

    @Nullable
    private DataInputStream createExternalChunkInputStream(ChunkPos chunkcoordintpair, byte b0) throws IOException {
        Path path = this.getExternalChunkPath(chunkcoordintpair);
        if (!Files.isRegularFile(path, new LinkOption[0])) {
            LOGGER.error("External chunk path {} is not file", (Object)path);
            return null;
        }
        return this.createChunkInputStream(chunkcoordintpair, b0, Files.newInputStream(path, new OpenOption[0]));
    }

    private static ByteArrayInputStream createStream(ByteBuffer bytebuffer, int i) {
        return new ByteArrayInputStream(bytebuffer.array(), bytebuffer.position(), i);
    }

    private int packSectorOffset(int i, int j) {
        return i << 8 | j;
    }

    private static int getNumSectors(int i) {
        return i & 0xFF;
    }

    private static int getSectorNumber(int i) {
        return i >> 8 & 0xFFFFFF;
    }

    private static int sizeToSectors(int i) {
        return (i + 4096 - 1) / 4096;
    }

    public boolean doesChunkExist(ChunkPos chunkcoordintpair) {
        int i = this.getOffset(chunkcoordintpair);
        if (i == 0) {
            return false;
        }
        int j = RegionFile.getSectorNumber(i);
        int k = RegionFile.getNumSectors(i);
        ByteBuffer bytebuffer = ByteBuffer.allocate(5);
        try {
            this.file.read(bytebuffer, j * 4096);
            ((Buffer)bytebuffer).flip();
            if (bytebuffer.remaining() != 5) {
                return false;
            }
            int l = bytebuffer.getInt();
            byte b0 = bytebuffer.get();
            if (RegionFile.isExternalStreamChunk(b0)) {
                if (!RegionFileVersion.isValidVersion(RegionFile.getExternalChunkVersion(b0))) {
                    return false;
                }
                if (!Files.isRegularFile(this.getExternalChunkPath(chunkcoordintpair), new LinkOption[0])) {
                    return false;
                }
            } else {
                if (!RegionFileVersion.isValidVersion(b0)) {
                    return false;
                }
                if (l == 0) {
                    return false;
                }
                int i1 = l - 1;
                if (i1 < 0 || i1 > 4096 * k) {
                    return false;
                }
            }
            return true;
        }
        catch (IOException ioexception) {
            return false;
        }
    }

    public DataOutputStream getChunkDataOutputStream(ChunkPos chunkcoordintpair) throws IOException {
        return new DataOutputStream(this.version.wrap(new ChunkBuffer(chunkcoordintpair)));
    }

    public void flush() throws IOException {
        this.file.force(true);
    }

    public void clear(ChunkPos chunkcoordintpair) throws IOException {
        int i = RegionFile.getOffsetIndex(chunkcoordintpair);
        int j = this.offsets.get(i);
        if (j != 0) {
            this.offsets.put(i, 0);
            this.timestamps.put(i, RegionFile.getTimestamp());
            this.writeHeader();
            Files.deleteIfExists(this.getExternalChunkPath(chunkcoordintpair));
            this.usedSectors.free(RegionFile.getSectorNumber(j), RegionFile.getNumSectors(j));
        }
    }

    protected synchronized void write(ChunkPos chunkcoordintpair, ByteBuffer bytebuffer) throws IOException {
        CommitOp regionfile_b;
        int k1;
        int i = RegionFile.getOffsetIndex(chunkcoordintpair);
        int j = this.offsets.get(i);
        int k = RegionFile.getSectorNumber(j);
        int l = RegionFile.getNumSectors(j);
        int i1 = bytebuffer.remaining();
        int j1 = RegionFile.sizeToSectors(i1);
        if (j1 >= 256) {
            Path path = this.getExternalChunkPath(chunkcoordintpair);
            LOGGER.warn("Saving oversized chunk {} ({} bytes} to external file {}", new Object[]{chunkcoordintpair, i1, path});
            j1 = 1;
            k1 = this.usedSectors.allocate(j1);
            regionfile_b = this.writeToExternalFile(path, bytebuffer);
            ByteBuffer bytebuffer1 = this.createExternalStub();
            this.file.write(bytebuffer1, k1 * 4096);
        } else {
            k1 = this.usedSectors.allocate(j1);
            regionfile_b = () -> Files.deleteIfExists(this.getExternalChunkPath(chunkcoordintpair));
            this.file.write(bytebuffer, k1 * 4096);
        }
        this.offsets.put(i, this.packSectorOffset(k1, j1));
        this.timestamps.put(i, RegionFile.getTimestamp());
        this.writeHeader();
        regionfile_b.run();
        if (k != 0) {
            this.usedSectors.free(k, l);
        }
    }

    private ByteBuffer createExternalStub() {
        ByteBuffer bytebuffer = ByteBuffer.allocate(5);
        bytebuffer.putInt(1);
        bytebuffer.put((byte)(this.version.getId() | 0x80));
        ((Buffer)bytebuffer).flip();
        return bytebuffer;
    }

    private CommitOp writeToExternalFile(Path path, ByteBuffer bytebuffer) throws IOException {
        Path path1 = Files.createTempFile(this.externalFileDir, "tmp", (String)null, new FileAttribute[0]);
        try (FileChannel filechannel = FileChannel.open(path1, StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            ((Buffer)bytebuffer).position(5);
            filechannel.write(bytebuffer);
        }
        return () -> Files.move(path1, path, StandardCopyOption.REPLACE_EXISTING);
    }

    private void writeHeader() throws IOException {
        ((Buffer)this.header).position(0);
        this.file.write(this.header, 0L);
    }

    private int getOffset(ChunkPos chunkcoordintpair) {
        return this.offsets.get(RegionFile.getOffsetIndex(chunkcoordintpair));
    }

    public boolean hasChunk(ChunkPos chunkcoordintpair) {
        return this.getOffset(chunkcoordintpair) != 0;
    }

    private static int getOffsetIndex(ChunkPos chunkcoordintpair) {
        return chunkcoordintpair.getRegionLocalX() + chunkcoordintpair.getRegionLocalZ() * 32;
    }

    @Override
    public void close() throws IOException {
        try {
            this.padToFullSector();
        }
        finally {
            try {
                this.file.force(true);
            }
            finally {
                this.file.close();
            }
        }
    }

    private void padToFullSector() throws IOException {
        int j;
        int i = (int)this.file.size();
        if (i != (j = RegionFile.sizeToSectors(i) * 4096)) {
            ByteBuffer bytebuffer = PADDING_BUFFER.duplicate();
            ((Buffer)bytebuffer).position(0);
            this.file.write(bytebuffer, j - 1);
        }
    }

    private class ChunkBuffer
    extends ByteArrayOutputStream {
        private final ChunkPos pos;

        public ChunkBuffer(ChunkPos chunkcoordintpair) {
            super(8096);
            super.write(0);
            super.write(0);
            super.write(0);
            super.write(0);
            super.write(RegionFile.this.version.getId());
            this.pos = chunkcoordintpair;
        }

        @Override
        public void close() throws IOException {
            ByteBuffer bytebuffer = ByteBuffer.wrap(this.buf, 0, this.count);
            int i = this.count - 5 + 1;
            JvmProfiler.INSTANCE.onRegionFileWrite(RegionFile.this.info, this.pos, RegionFile.this.version, i);
            bytebuffer.putInt(0, i);
            RegionFile.this.write(this.pos, bytebuffer);
        }
    }

    private static interface CommitOp {
        public void run() throws IOException;
    }
}

